

import graphtools.GraphAlgorithmsProxy;
import graphtools.GraphConverterRequest;
import graphtools.GraphConverterResponse;
import graphtools.MetabolicGraphConstructorRequest;
import graphtools.MetabolicGraphConstructorResponse;
import graphtools.MixedRequest;
import graphtools.PathfinderRequest;
import graphtools.PathfinderResponse;
import graphtools.PathwayinferenceRequest;
import graphtools.WorkFlowRequest;
import graphtools.WorkFlowResponse;

import java.io.File;
import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.Scanner;
import java.util.Set;

import RSATWS.CompareGraphsRequest;
import RSATWS.CompareGraphsResponse;
import RSATWS.GraphNodeDegreeRequest;
import RSATWS.GraphNodeDegreeResponse;
import RSATWS.RSATWSPortTypeProxy;
import be.ac.ulb.bigre.pathwayinference.core.PathwayinferenceConstants;
import be.ac.ulb.bigre.pathwayinference.core.io.IOTools;
import be.ac.ulb.bigre.pathwayinference.core.util.DiverseTools;
import be.ac.ulb.bigre.pathwayinference.core.util.Groups;


/**
 * The example client demonstrates how to call NeAT
 * tools programmatically via SOAP.
 *
 * @author Karoline Faust
 *
 */
public class GraphAlgorithmsClient  {

	public static String GraphAlgorithms_address = "http://wwwsup.scmbb.ulb.ac.be/be.ac.ulb.bigre.graphtools.server/services/GraphAlgorithms";

	/**
	 * Do a demo path finding and graph conversion.
	 * Transfer data on client side.
	 * Finally, do a graph comparison.
	 */
	public static void demo(){
		try
		{
			/* Get the location of the service */
			GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
			proxy.setEndpoint(GraphAlgorithms_address);

			/* prepare the path finding parameters */
			PathfinderRequest paramsPathfinder = new PathfinderRequest();

			// source and targets
			String source = "a";
			String target = "w";
			// very simple graph
			String simpleGraphString = "a	b\nb	c\nc	d\nc	f\nc	w\nd	w\n";
			paramsPathfinder.setSource(source);
			paramsPathfinder.setTarget(target);
			paramsPathfinder.setGraphString(simpleGraphString);
			paramsPathfinder.setReturnType("client");
			paramsPathfinder.setOutputType("pathsUnion");
			paramsPathfinder.setOutFormat("gml");

			/* Call the pathfinding service */
			System.out.println("Call pathfinding service...");
			PathfinderResponse pathfinderResult = proxy.pathfinding(paramsPathfinder);

			/* Process path finding results  */
			//Report the location on the server (empty if returnType equals client)
			System.out.println("Location of result on the server:\n"+ pathfinderResult.getServer());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ pathfinderResult.getCommand());
			// Report the graph id
			System.out.println("Graph id:\n"+ pathfinderResult.getGraphid());
			// Report result
			System.out.println("Result:\n"+ pathfinderResult.getClient());

			GraphConverterRequest paramsGraphConverter = new GraphConverterRequest();
			paramsGraphConverter.setGraphString(pathfinderResult.getClient());
			paramsGraphConverter.setInFormat(paramsPathfinder.getOutFormat());
			paramsGraphConverter.setOutFormat("visml");
			paramsGraphConverter.setReturnType("client");

			/* Call the graph conversion service */
			System.out.println("Call graph conversion service...");
			GraphConverterResponse graphconversionResult = proxy.graphconversion(paramsGraphConverter);

			/* Process graphconversion results */
			// Report the location on the server (empty if returnType equals client)
			System.out.println("Location of result on the server:\n"+ graphconversionResult.getServer());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ graphconversionResult.getCommand());
			// Report result
			System.out.println("Result:\n"+ graphconversionResult.getClient());

			String anotherSimpleGraphString = "a	q\nq	b\ng	q\nh	b\n";

		 	/* Get the location of the RSAT service */
		 	RSATWSPortTypeProxy RSATproxy = new RSATWSPortTypeProxy();

		 	/* formulate a graph comparison request */
		 	CompareGraphsRequest compareGraphsRequest = new CompareGraphsRequest();
		 	compareGraphsRequest.setQinputgraph(simpleGraphString);
		 	compareGraphsRequest.setQinformat("tab");
		 	compareGraphsRequest.setQscol(1);
		 	compareGraphsRequest.setQtcol(2);
		 	compareGraphsRequest.setRinputgraph(anotherSimpleGraphString);
		 	compareGraphsRequest.setRinformat("tab");
		 	compareGraphsRequest.setRscol(1);
		 	compareGraphsRequest.setRtcol(2);
		 	// compute intersection
		 	compareGraphsRequest.set_return("intersection");

			/* Call the graph comparison service */
		 	System.out.println("Call graph comparison service...");
		 	CompareGraphsResponse compareGraphsResponse = RSATproxy.compare_graphs(compareGraphsRequest);

		 	/* Process graph comparison results */
			// Report the location on the server
			System.out.println("Location of result on the server:\n"+ compareGraphsResponse.getServer());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ compareGraphsResponse.getCommand());
			// Report result
			System.out.println("Result:\n"+ compareGraphsResponse.getClient());


		}
		catch(Exception e) {
			System.out.println(e.toString());
		}
	}

	/**
	 * Do a demo path finding and graph conversion.
	 * Transfer data on server side.
	 */
	public static void demoServerSide(){
		try
		{
			/* Get the location of the service */
			GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
			proxy.setEndpoint(GraphAlgorithms_address);

			/* prepare the path finding parameters */
			PathfinderRequest paramsPathfinder = new PathfinderRequest();

			// source and targets
			String source = "a";
			String target = "w";
			// very simple graph
			String simpleGraphString = "a	b\nb	c\nc	d\nc	f\nc	w\nd	w\n";
			paramsPathfinder.setSource(source);
			paramsPathfinder.setTarget(target);
			paramsPathfinder.setGraphString(simpleGraphString);
			paramsPathfinder.setReturnType("server");
			paramsPathfinder.setOutputType("pathsUnion");
			paramsPathfinder.setOutFormat("gml");

			/* Call the pathfinding service */
			System.out.println("Call pathfinding service...");
			PathfinderResponse pathfinderResult = proxy.pathfinding(paramsPathfinder);

			/* Process path finding results  */
			//Report the location on the server (empty if returnType equals client)
			System.out.println("Location of result on the server:\n"+ pathfinderResult.getServer());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ pathfinderResult.getCommand());
			// Report the graph id
			System.out.println("Graph id:\n"+ pathfinderResult.getGraphid());

			GraphConverterRequest paramsGraphConverter = new GraphConverterRequest();
			paramsGraphConverter.setTmpInGraphFile(pathfinderResult.getServer());
			paramsGraphConverter.setInFormat(paramsPathfinder.getOutFormat());
			paramsGraphConverter.setOutFormat("visml");
			paramsGraphConverter.setReturnType("client");

			/* Call the graph conversion service */
			System.out.println("Call graph conversion service...");
			GraphConverterResponse graphconversionResult = proxy.graphconversion(paramsGraphConverter);

			/* Process graphconversion results */
			// Report the location on the server (empty if returnType equals client)
			System.out.println("Location of result on the server:\n"+ graphconversionResult.getServer());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ graphconversionResult.getCommand());
			// Report result
			System.out.println("Result:\n"+ graphconversionResult.getClient());
		}
		catch(Exception e) {
			System.out.println(e.toString());
		}
	}

    /**
     * This method calls Pathfinder on one of the pre-loaded metabolic networks
     * (the RPAIR network consisting of reactant pairs and compounds). It demonstrates
     * how to do metabolic path finding via the path finding web service. For the full list
     * of metabolic network identifiers, see the metabolic pathfinder manual page.
     */
    public static void demoMetabolicPathfinding(){
    	/* Get the location of the service */
		GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
		proxy.setEndpoint(GraphAlgorithms_address);

		/* prepare the path finding parameters */
		PathfinderRequest paramsPathfinder = new PathfinderRequest();

		// source and targets

		// Phospho-L-aspartate
		String source = "C03082";
		// L-Lysine
		String target = "C00047";

		// RPAIR network id
		String graphId = "Pathfinder_tmpGraph_b299d55c-a3b2-497b-bda8-e6608646e639.tab";

		paramsPathfinder.setSource(source);
		paramsPathfinder.setTarget(target);
		paramsPathfinder.setGraphString(graphId);
		// reactant pairs are weighted according to their class, compounds according to their degree
		paramsPathfinder.setWeight("rpairs");
		paramsPathfinder.setMetabolic(true);
		paramsPathfinder.setTmpInGraphFile("");
		// RPAIR graph is undirected
		paramsPathfinder.setDirected(false);
		// default exclusion attribute
		paramsPathfinder.setExclusionAttr("ExclusionAttribute");
		paramsPathfinder.setRank(5);
		paramsPathfinder.setMaxWeight(5000);
		paramsPathfinder.setMaxLength(50);
		paramsPathfinder.setMinLength(1);
		paramsPathfinder.setInFormat("flat");
		paramsPathfinder.setReturnType("client");
		paramsPathfinder.setOutputType("pathsUnion");
		paramsPathfinder.setOutFormat("flat");

		/* Call the pathfinding service */
		System.out.println("Call pathfinding service...");
		PathfinderResponse pathfinderResult;
		try {
			pathfinderResult = proxy.pathfinding(paramsPathfinder);

			//Report the result
			System.out.println("Result:\n"+ pathfinderResult.getClient());
			//Report the remote command
			System.out.println("Command used on the server:\n"+ pathfinderResult.getCommand());
			// Report the graph id
			System.out.println("Graph id:\n"+ pathfinderResult.getGraphid());
		} catch (RemoteException e) {
			e.printStackTrace();
		}
    }


    /**
     * This demo shows how to construct a metabolic graph and
     * afterwards calculate its node degree distribution.
     */
    public static void demoMetabolicGraphConstructionAndNodeDegree(){
    	/* Get the location of the service */
		GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
		proxy.setEndpoint(GraphAlgorithms_address);

		String format = "tab";
		String nodeDegreeInputGraph = "";

		/* formulate the graph construction request */
		MetabolicGraphConstructorRequest construction = new MetabolicGraphConstructorRequest();
		construction.setDirected(false);
		// merge the metabolic graphs from three Saccharomyces species
		construction.setOrganismNames("sce/spo/dsba");
		construction.setGraphType("r");
		construction.setOutFormat(format);
		construction.setReturnType("client");

		/* Call the service */
		System.out.println("Call metabolicgraphconstruction service...");
		MetabolicGraphConstructorResponse res;
		try {
			res = proxy.metabolicgraphconstruction(construction);
			nodeDegreeInputGraph = res.getClient();
		} catch (RemoteException e) {
			e.printStackTrace();
		}

		/** remove nodes and their annotation (cannot be processed by node degree topology, edges/arcs are sufficient) */
		nodeDegreeInputGraph = nodeDegreeInputGraph.split(";ARCS")[1];

	 	/* Get the location of the RSAT service */
	 	RSATWSPortTypeProxy RSATproxy = new RSATWSPortTypeProxy();
	 	/* formulate a graph node degree request */
	 	GraphNodeDegreeRequest nodeDegreeRequest = new GraphNodeDegreeRequest();
	 	nodeDegreeRequest.setInformat(format);
	 	nodeDegreeRequest.setAll(1);
	 	nodeDegreeRequest.setInputgraph(nodeDegreeInputGraph);
	 	nodeDegreeRequest.setScol(1);
	 	nodeDegreeRequest.setTcol(2);


	 	try{
	 		/* Call the graph comparison service */
	 		System.out.println("Call node degree computation service...");
	 		GraphNodeDegreeResponse nodeDegreeResponse = RSATproxy.graph_node_degree(nodeDegreeRequest);

	 		/* Process graph comparison results */
	 		// Report the location on the server
	 		System.out.println("Location of result on the server:\n"+ nodeDegreeResponse.getServer());
	 		//Report the remote command
	 		System.out.println("Command used on the server:\n"+ nodeDegreeResponse.getCommand());
	 		// Report result
	 		System.out.println("Result:\n"+ nodeDegreeResponse.getClient());
	 	}catch(RemoteException e){
	 		e.printStackTrace();
	 	}
    }

	/**
	 * This method constructs first the metabolic RPAIR network of
	 * S. cerevisiae and launches afterwards a pathway inference task
	 * on it.
	 * You can specify your email address to receive the results via email.
	 */
	public static void demoPathwayinferenceWorkflow(){
		/* Get the location of the service */
		GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
		proxy.setEndpoint(GraphAlgorithms_address);

		boolean directed = true;

		/* formulate the graph construction request */
		MetabolicGraphConstructorRequest construction = new MetabolicGraphConstructorRequest();
		construction.setDirected(directed);
		construction.setOrganismNames("sce");
		construction.setGraphType("s");
		construction.setExcludeRPairClasses("leave/ligase");

		// seeds
		String seeds = "C00075#C00024#C00092";

		/* prepare the parameters */
		PathwayinferenceRequest inference = new PathwayinferenceRequest();
		inference.setSeeds(seeds);
		inference.setDirected(directed);
		inference.setMetabolic(true);
		inference.setExclusionAttr("ExclusionAttribute");
		inference.setWeight("con");
		inference.setAlgorithm("takahashi");
		inference.setInFormat("flat");
		inference.setOutFormat("gml");
		inference.setOutputType("pathsUnion");

		/**
		 * WorkFlow
		 */
		WorkFlowRequest workflowReq = new WorkFlowRequest();
		// workflowReq.setEmail("your email");

		MixedRequest[] requestArray = new MixedRequest[2];
		MixedRequest mixedReq1 = new MixedRequest();
		MixedRequest mixedReq2 = new MixedRequest();
		mixedReq1.setMetabolicGraphConstructorRequest(construction);
		mixedReq2.setPathwayinferenceRequest(inference);
		requestArray[0] = mixedReq1;
		requestArray[1] = mixedReq2;

		workflowReq.setRequestArray(requestArray);
		workflowReq.setReturnType("client");

		/* Call the service */
		System.out.println("Call graphtools server...");
		WorkFlowResponse res;
		try {
			res = proxy.workflow(workflowReq);
			System.out.println("Location of result on the server:\n"+res.getServer());
			System.out.println("Command used on the server:\n"+res.getCommand());
			System.out.println("Result:\n"+res.getClient());
		} catch (RemoteException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Find paths in a pre-loaded gene network using a batch file.
	 */
	public static void pathfindingGenePairs(){
		/* variables */
		// output format (tab or gml)
		String graphOutputFormat = "gml";
		String graphSuffix = ".gml";
		// location of batch file
		String batchFileLocation = "";
		// identifier of pre-loaded graph (graph has been stored on server)
		String graphId = "Pathfinder_tmpGraph_a3eff75a-f7a8-48db-b5a2-5df96f8724f7.tab";
		// result file name
		String resultName = "Pathfinding_Result_GenePairs"+graphSuffix;
		// export folder
		String exportFolder = "Results";
		// export tmp results
		boolean exportTmp = true;

		/* fields */
		Groups experiments = new Groups();
		HashSet<String> startAndEnds = new HashSet<String>();
		String startGroupId = "";
		String start = "";
		String endGroupId = "";
		String end = "";
		boolean seedsOK = true;
		boolean firstRound = true;
		String mergedNetwork = "";
		String failure = "pathfinding failed";

		// create export folder in the current directory if it doesn't exist yet
		File exportFolderFile = new File(exportFolder);
		if(!exportFolderFile.exists() && exportTmp) exportFolderFile.mkdirs();

		// read in location of batch file provided by user
		System.out.println("Indicate location of batch file (full path, e.g. /home/username/batchfile.txt) and press Enter.");
		Scanner scanner = new Scanner(System.in);
		batchFileLocation = scanner.nextLine();
		scanner.close();

		try{
			Long startTime = System.currentTimeMillis();
			/* Get the location of the service */
			GraphAlgorithmsProxy proxy = new GraphAlgorithmsProxy();
			proxy.setEndpoint(GraphAlgorithms_address);
		 	/* Get the location of the RSAT service */
		 	RSATWSPortTypeProxy RSATproxy = new RSATWSPortTypeProxy();

		 	/* prepare query and response objects */

		 	// path finding
			PathfinderRequest paramsPathfinder = new PathfinderRequest();
			paramsPathfinder.setGraphString(graphId);
			paramsPathfinder.setDirected(false);
			paramsPathfinder.setWeight("unit");
			paramsPathfinder.setOutputType("pathsUnion");
			paramsPathfinder.setOutFormat(graphOutputFormat);
			paramsPathfinder.setReturnType("client");
			PathfinderResponse pathfinderResult = new PathfinderResponse();

			// graph union
			CompareGraphsRequest compareGraphsRequest = new CompareGraphsRequest();
			compareGraphsRequest.setQinformat(graphOutputFormat);
			compareGraphsRequest.setQscol(1);
			compareGraphsRequest.setQtcol(2);
			compareGraphsRequest.setRinformat(graphOutputFormat);
			compareGraphsRequest.setRscol(1);
			compareGraphsRequest.setRtcol(2);
			compareGraphsRequest.setOutformat(graphOutputFormat);
			compareGraphsRequest.set_return("union");
			CompareGraphsResponse compareGraphsResponse = new CompareGraphsResponse();

			/* load batch file */
			experiments.parseGroupsFile(batchFileLocation, false);
			Set<String> experimentIdentifiers = experiments.getLayer(0);

			// loop over experiments
			for(String expId : experimentIdentifiers){
				// get start and end node group identifiers of current experiment
				startAndEnds = experiments.getMembersOfGivenLayerAndSuperGroup(1,expId);
				for(String groupId : startAndEnds){
					if(groupId.contains(PathwayinferenceConstants.START_NODE)) startGroupId = groupId;
					if(groupId.contains(PathwayinferenceConstants.END_NODE)) endGroupId = groupId;
				}
				// do some checks
				if(startAndEnds.size() > 2){
					System.err.println("Warning: Experiment "+expId+" has more than one start group and one end group!");
					seedsOK = false;
				}
				if(startGroupId.equals("")){
					System.err.println("Warning: No start node group specified for experiment "+expId+"!");
					seedsOK = false;
				}
				if(endGroupId.equals("")){
					System.err.println("Warning: No end node group specified for experiment "+expId+"!");
					seedsOK = false;
				}
				if(experiments.getMembersOfGroup(startGroupId).size() < 1){
					System.err.println("Warning: Less than one start node specified for experiment "+expId+"!");
					seedsOK = false;
				}
				if(experiments.getMembersOfGroup(endGroupId).size() < 1){
					System.err.println("Warning: Less than one end node specified for experiment "+expId+"!");
					seedsOK = false;
				}
				// obtain start and end node groups of current experiment
				start = DiverseTools.setToString(experiments.getMembersOfGroup(startGroupId), "/");
				end = DiverseTools.setToString(experiments.getMembersOfGroup(endGroupId), "/");
				// print info on current experiment
				System.out.println("Processing experiment: "+expId);
				System.out.println("Start nodes: "+start);
				System.out.println("End nodes: "+end);

				if(seedsOK){
					// update pathfinding request
					paramsPathfinder.setSource(start);
					paramsPathfinder.setTarget(end);
					// call pathfinding web service
					pathfinderResult = proxy.pathfinding(paramsPathfinder);
					// save temp result into a file
					if(exportTmp){
						IOTools.exportStringToFile(pathfinderResult.getClient(),exportFolder+File.separator+expId+"_PathfinderResult"+graphSuffix);
						System.out.println("Info: Result for experiment "+expId+" has been saved in folder "+exportFolder+".");
					}
					if(!pathfinderResult.getClient().contains(failure)){
						if(!firstRound){
							// update graph comparison request
							compareGraphsRequest.setQinputgraph(pathfinderResult.getClient());
							compareGraphsRequest.setRinputgraph(mergedNetwork);
							// call graph comparison web service
							compareGraphsResponse = RSATproxy.compare_graphs(compareGraphsRequest);
							// update merged network
							mergedNetwork = compareGraphsResponse.getClient();
						}else{
							mergedNetwork = pathfinderResult.getClient();
							firstRound = false;
						}
					// path finding failed
					}else{
						System.err.println("Warning: Experiment "+expId+" failed!");
					}
				// seed nodes problematic
				}else{
					System.err.println("Problem with seed nodes. Skipping experiment "+expId+"!");
				}
				// reset fields
				seedsOK = true;
				startGroupId = "";
				start = "";
				endGroupId = "";
				end = "";
			} // end loop over experiments

			// save unified network
			IOTools.exportStringToFile(mergedNetwork, resultName);
			// notify user
			System.out.println("Info: A file with name "+resultName+" containing the results has been created in the current directory.");
			Long endTime = System.currentTimeMillis();
			System.out.println("Info: Computation of k-shortest paths took "+(endTime-startTime)+" ms.");

		}
		catch(Exception e) {
			System.out.println(e.toString());
		}
	}


	/**
	 * Main.
	 * @param args
	 */
	public static void main(String[] args){
		demo();
		//demoServerSide();
		//demoMetabolicPathfinding();
		//demoMetabolicGraphConstructionAndNodeDegree();
		//demoPathwayinferenceWorkflow();
		//pathfindingGenePairs();
	}

}