################################################################
## Prepare a release of RSAT

include ${RSAT}/makefiles/util.mk
MAKEFILE=${RSAT}/makefiles/release_rsat.mk
MAKE = make -sk -f ${MAKEFILE}

## Archive file
DATE=`date +%Y.%m.%d`
ARCHIVE_PREFIX=rsat_${DATE}
ARCHIVE_PREFIX_CORE=rsat-core_${DATE}
ARCHIVE_PREFIX_WEB=rsat-web_${DATE}
ARCHIVE_PREFIX_MOTIFDB=rsat-motifdb_${DATE}
ARCHIVE_PREFIX_METAB=metabolic-tools_${DATE}
ARCHIVE=rsat/${ARCHIVE_PREFIX}
ARCHIVE_PREFIX_SCRIPTS=${ARCHIVE_PREFIX}_install_scripts

## Archive with tar
#TAR_EXCLUDE=-X CVS '*~' 
TAR_EXCLUDE=--exclude .git \
	--exclude CVS \
	--exclude '*~~' \
	--exclude tmp \
	--exclude data \
	--exclude logs \
	--exclude doc/manuals/*.toc \
	--exclude doc/manuals/*.idx \
	--exclude doc/manuals/*.aux \
	--exclude doc/manuals/*.log \
	--exclude doc/manuals/*.out \
	--exclude course \
	--exclude eccb_2014_tutorial \
	--exclude '*.o' \
	--exclude peak-footprints \
	--exclude floydwarshall \
	--exclude kwalks \
	--exclude REA \
	--exclude REA-* \
	--exclude purgatory_* \
	--exclude .Rproj.user \
	--exclude '*.RData' \
	--exclude Rpackages \
	--exclude purgatory
TAR_CREATE=tar ${TAR_EXCLUDE} ${TAR_OPT} -cpf ${ARCHIVE}.tar rsat/*_default.*
TAR_APPEND=tar ${TAR_EXCLUDE} ${TAR_OPT} -rpf ${ARCHIVE}.tar 

################################################################
## All the tasks for publishing the new version
#all: manuals tar_archive clean_release_site publish
all: manuals tar_archive_rsat publish_rsat

## List parameters
#PUB_SERVER=rsat.ulb.ac.be
PUB_SERVER=pedagogix-tagc.univ-mrs.fr
PUB_LOGIN=rsat
SSH_OPT=
PUB_FORMAT=tar.gz
PUB_DIR=/data/rsat_release
RELEASE_URL=http://${PUB_SERVER}/download_rsat/
list_param:
	@echo "RSAT release parameters"
	@echo "	ARCHIVE			${ARCHIVE}"
	@echo "	PUB_LOGIN		${PUB_LOGIN}"
	@echo "	PUB_SERVER		${PUB_SERVER}"
	@echo "	PUB_DIR			${PUB_DIR}"
	@echo "	RELEASE_URL		${RELEASE_URL}"
	@echo "	PUB_TARGET_MANUALS	${PUB_TARGET_MANUALS}"

################################################################
## Generate the Manuals and tutorials
manuals:
	@echo "Generating installation manuals and user guides"
	(cd doc/manuals; make fullclean; make all_pdf; make tex_clean)
	mkdir -p public_html/release
	rsync -rtupvl -e "ssh ${SSH_OPT}" doc/manuals/*.pdf public_html/release
	@echo "Up-to-date manuals can be found in public_html/release" 

## Install manuals on the RSAT Web server
PUB_TARGET_MANUALS=${PUB_LOGIN}@${PUB_SERVER}:rsat/public_html/release/
publish_manuals:
	echo "Publishing manuals on server ${PUB_TARGET_MANUALS}"
	rsync -ruptvl -e 'ssh ${SSH_OPT}' doc/manuals/*.pdf ${PUB_TARGET_MANUALS}

################################################################
## Clean temporary backup files generated by emacs
CLEAN_DIRS=perl-scripts makefiles R-scripts doc ws_clients public_html/tutorials public_html/web_services 
clean_emacs_bk:
	@for dir in ${CLEAN_DIRS}; do \
		echo "	Cleaning directory" $${dir} ; \
		find $${dir} -name '*~' -exec rm {} \; ; \
		find $${dir} -name '.#*' -exec rm {} \; ; \
		find $${dir} -name '#*' -exec rm {} \; ; \
	done
	@echo "	Cleaning directory public_html"
	@rm -f public_html/*~ public_html/.#* public_html/#*
	@rm -f *~ .#* #*

################################################################
## Create tar and zip archives of the whole release
POST_CMD=
TAR_BASE=`dirname ${RSAT}`
RSAT_CORE=rsat/00_README.txt			\
	rsat/LICENSE.txt			\
	rsat/INSTALL.md				\
	rsat/bin/rsat				\
	rsat/share				\
	rsat/installer				\
	rsat/perl-scripts			\
	rsat/makefiles				\
	rsat/RSAT_config_default.props		\
	rsat/RSAT_config_default.mk		\
	rsat/RSAT_config_default.bashrc		\
	rsat/RSAT_config_default.conf		\
	rsat/RSAT_config_default_apache2.2.conf	\
	rsat/doc/manuals			\
	rsat/doc/howto				\
	rsat/python-scripts 			\
	rsat/contrib/count-words  		\
	rsat/contrib/compare-matrices-quick  	\
	rsat/contrib/info-gibbs			\
	rsat/contrib/matrix-scan-quick  	\
	rsat/contrib/variation-scan	  	\
	rsat/contrib/retrieve-variation-seq  	\
	rsat/ws_clients		  		\
	rsat/R-scripts

RSAT_FILES=${RSAT_CORE} ${RSAT_WEB}

RSAT_WEB=rsat/public_html

MOTIFDB=rsat/public_html/motif_databases

#	rsat/R-scripts/TFBMclust		\
#	rsat/R-scripts/*.R			\
#	rsat/R-scripts/util			\

PATHWAY_FILES = \
	rsat/contrib/floydwarshall		\
	rsat/contrib/REA			\
	rsat/contrib/kwalks

RSAT_FILES_METAB=rsat/java		\
	rsat/contrib/REA		\
	rsat/contrib/kwalks

#RSAT_FILES_SCRIPTS=rsat/installer

_create_tar_archive:
	@echo "Creating tar archive ${ARCHIVE}.tar"
	@echo ${TAR_CREATE} 
	(cd ${TAR_BASE}; ${TAR_CREATE})

FILE=rsat/perl-scripts
_add_one_file:
	@echo ${ARCHIVE_CMD} ${FILE} ${POST_CMD}
	${ARCHIVE_CMD} ${FILE}  ${POST_CMD}

_fill_archive:
	(cd ${TAR_BASE};				\
	for f in ${RSAT_FILES}; do			\
		${MAKE} _add_one_file FILE=$${f};	\
	done)
	@echo "Archive created	${ARCHIVE}"

## Create an archive with RSAT/NeAT tools
MD5SUM=md5sum
MD5_FILE=${TAR_BASE}/${ARCHIVE}.tar.gz.md5
SHA256_FILE=${TAR_BASE}/${ARCHIVE}.tar.gz.sha256
#SHA256_FILE=${ARCHIVE}.tar.gz.sha256
SHASUM_CMD=shasum -a 256 ${ARCHIVE}.tar.gz  > ${SHA256_FILE}
tar_archive:
	@echo
	@echo "tar archive ${ARCHIVE_PREFIX_CORE}"
	@${MAKE} clean_emacs_bk
	@${MAKE} _create_tar_archive
	@${MAKE} _fill_archive ARCHIVE_CMD='${TAR_APPEND}' POST_CMD=''
	(cd ${TAR_BASE}; gzip -f ${ARCHIVE}.tar; ${SHASUM_CMD}; ${MD5SUM} ${ARCHIVE}.tar.gz > ${MD5_FILE})
	@echo
	@echo "Archive		${TAR_BASE}/${ARCHIVE}.tar.gz"
	@echo "md5 file		${MD5_FILE}"
	@cat ${MD5_FILE}
	@echo "sha256 file	${SHA256_FILE}"
	@cat ${SHA256_FILE}

## Create an archive with the command-line tools only (no web site, no data)
tar_archive_core:
	${MAKE} tar_archive ARCHIVE_PREFIX=${ARCHIVE_PREFIX_CORE} RSAT_FILES="${RSAT_CORE}" TAR_OPT='--exclude ${MOTIFDB}'

## Create an archive with the web site (+ demo data but exlude the motif databases)
tar_archive_web:
	${MAKE} tar_archive ARCHIVE_PREFIX=${ARCHIVE_PREFIX_WEB} RSAT_FILES="${RSAT_WEB}" TAR_OPT='--exclude ${MOTIFDB}'

## Create an archive with the motif datavases
tar_archive_motifdb:
	${MAKE} tar_archive ARCHIVE_PREFIX=${ARCHIVE_PREFIX_MOTIFDB} RSAT_FILES="${MOTIFDB}"

## Create the separate RSAT archives
tar_archive_rsat: tar_archive_core tar_archive_web tar_archive_motifdb

## Create an archive with the metabolic tools (since the java files occupy 80Mb, we releaseute them separately
tar_archive_metab:
	${MAKE} tar_archive ARCHIVE_PREFIX=${ARCHIVE_PREFIX_METAB} RSAT_FILES="${RSAT_FILES_METAB}"

#tar_archive_scripts:
#	${MAKE} tar_archive ARCHIVE_PREFIX=${ARCHIVE_PREFIX_SCRIPTS} RSAT_FILES="${RSAT_FILES_SCRIPTS}"

## Archive with zip
# ZIP_EXCLUDE=-x CVS '*~' tmp data logs
# ZIP =zip -ry ${ARCHIVE}.zip 
# zip_archive:
# 	${MAKE} _fill_archive ARCHIVE_CMD='${ZIP}' POST_CMD='${ZIP_EXCLUDE}'


ls_release_site:
	ssh ${SSH_OPT} ${PUB_LOGIN}@${PUB_SERVER} "ls -ltra ${PUB_DIR}/"

clean_release_site:
	@echo
#	@echo "Moving previous archives from the public server ${PUB_LOGIN}@${PUB_SERVER} to ${PUB_DIR}/previous_versions"
#	ssh ${SSH_OPT} ${PUB_LOGIN}@${PUB_SERVER} "mv -f ${PUB_DIR}/rsat_*.tar.gz ${PUB_DIR}/previous_versions/"
	@echo 
	@echo "BEWARE: the tar archives for RSAT code and install scripts have been moved to previous_version folder. "
	@echo "Do not forget to publish a new version with"
	@echo "	make -f makefiles/rsat_release.mk publish"

################################################################
## Publish the tar archive of the whole release
publish:
	@echo
	@echo "Synchronizing RSAT archive ${ARCHIVE_PREFIX}.${PUB_FORMAT} to server ${PUB_LOGIN}@${PUB_SERVER}:${PUB_DIR}"
	@echo
	rsync -ruptvl -e "ssh ${SSH_OPT}" ${ARCHIVE_PREFIX}.${PUB_FORMAT} ${PUB_LOGIN}@${PUB_SERVER}:${PUB_DIR}/
	rsync -ruptvl -e "ssh ${SSH_OPT}" ${ARCHIVE_PREFIX}.${PUB_FORMAT}.md5 ${PUB_LOGIN}@${PUB_SERVER}:${PUB_DIR}/
	rsync -ruptvl -e "ssh ${SSH_OPT}" ${ARCHIVE_PREFIX}.${PUB_FORMAT}.sha256 ${PUB_LOGIN}@${PUB_SERVER}:${PUB_DIR}/
	@ssh ${SSH_OPT} ${PUB_LOGIN}@${PUB_SERVER} "cd ${PUB_DIR}; ln -sf ${ARCHIVE_PREFIX}.${PUB_FORMAT} latest"
	@echo
	@echo "The archive should be accessible on the RSAT download server"	
	@echo "	http://download.rsat.eu/"

## Publish the tar archive with RSAT core
publish_core:
	@${MAKE} publish ARCHIVE_PREFIX=${ARCHIVE_PREFIX_CORE}

## Publish the tar archive with RSAT Web server
publish_web:
	@${MAKE} publish ARCHIVE_PREFIX=${ARCHIVE_PREFIX_WEB}

## Publish the tar archive with motif databases
publish_motifdb:
	@${MAKE} publish ARCHIVE_PREFIX=${ARCHIVE_PREFIX_MOTIFDB}

## Publish the 3 separate archives with RSAT distribution
publish_rsat: publish_core publish_web publish_motifdb

#publish_scripts:
#	@${MAKE} publish ARCHIVE_PREFIX=${ARCHIVE_PREFIX_SCRIPTS}


## Publish the appliance for RSAT VirtualBox Virtual Machine (VM)
APPLIANCE=rsat-vb-2018-05.ova
LOCAL_APPLIANCE_DIR=/no_backup/VirtualBox_VMs/RSAT-VM_00/appliances/
LOCAL_APPLIANCE=${LOCAL_APPLIANCE_DIR}/${APPLIANCE}
RSATVM_REPO=rsat@pedagogix-tagc.univ-mrs.fr:/data/rsat_release/virtual_machines/
RSATVM_TUTO=${RSAT}/doc/howto/RSAT-VM/RSAT-VM_tuto.html
publish_vm_param:
	@echo
	@echo "Parameters for RSAT-VM publication"
	@echo "	APPLIANCE		${APPLIANCE}"
	@echo "	LOCAL_APPLIANCE_DIR	${LOCAL_APPLIANCE_DIR}"
	@echo "	LOCAL_APPLIANCE		${LOCAL_APPLIANCE}"
	@echo "	Local appliance size (Mb)	`du -sm ${LOCAL_APPLIANCE}`"
	@echo "	RSATVM_REPO		${RSATVM_REPO}"
	@echo "	RSATVM_TUTO		${RSATVM_TUTO}"
	@echo "			${}"

publish_vm:
	@echo
	@echo "Synchronizing RSAT-VM	${APPLIANCE}"
	@echo "	LOCAL_APPLIANCE	${LOCAL_APPLIANCE}"
	@chmod 644 ${LOCAL_APPLIANCE}
	@du -sm ${LOCAL_APPLIANCE}
	@chmod 644 ${LOCAL_APPLIANCE}
	@rsync -ruptvl ${LOCAL_APPLIANCE}  ${RSATVM_REPO}

publish_vm_tuto:
	@echo
	@echo "Synchronizing RSAT-VM tutorial"
	@rsync -ruptvl ${RSATVM_TUTO}  ${RSATVM_REPO}
	@echo "Check the web site"
	@echo "	http://download.rsat.eu/virtual_machines/"
	@echo "	http://download.rsat.eu/virtual_machines/RSAT-VM_tuto.html"

publish_metab:
	@${MAKE} publish ARCHIVE_PREFIX=${ARCHIVE_PREFIX_METAB}

## Open the release Web site
BROWSER=firefox
web:
	 open -a ${BROWSER} ${RELEASE_URL}

################################################################
## Make a tar archive of the ws clients
TAR_WSCLIENTS=public_html/web_services/RSATWS_clients.tar.gz
tar_wsclients:
	@rm -f ${TAR_WSCLIENTS}
	tar \
	--exclude CVS  \
	--exclude '*~' \
	--exclude '*.DS_Store' -cpzf ${TAR_WSCLIENTS} ws_clients
	@echo ${TAR_WSCLIENTS}

publish_tar_wsclients:
	rsync -ruptvl -e "ssh ${SSH_OPT}" ${TAR_WSCLIENTS} ${PUB_LOGIN}@${PUB_SERVER}:rsat/public_html/web_services/

