#!flask/bin/python
from flask import Flask, jsonify, abort, request, make_response, url_for
import json
from subprocess import check_output, Popen, PIPE
import os,sys,re
import requests

app = Flask(__name__, static_url_path = "")

flask_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(flask_dir + '/lib')
import util

## Read environment variables
props_file = flask_dir + '/../../RSAT_config.props'
util.get_environ_vars(props_file)
os.environ['RSA_OUTPUT_CONTEXT'] = 'RSATWS'
rsat_home = os.environ['RSAT']
rsat_bin = os.environ['RSAT_BIN']
perl_scripts = rsat_home + '/perl-scripts'
public_html = rsat_home + '/public_html'

@app.errorhandler(400)
def not_found(error):
    return make_response(jsonify( { 'error': 'Bad request' } ), 400)
 
@app.errorhandler(404)
def not_found(error):
    return make_response(jsonify( { 'error': 'Not found' } ), 404)

@app.route('/', methods=['GET'])
def index():
    return "Hello World"

#### supported_organisms
@app.route('/supported-organisms', methods = ['POST', 'GET'])
def get_supported_organisms():
    output_choice = 'email'
    if request.method == 'POST':
        data = request.get_json(force=True) or request.form
    elif request.method == 'GET':
        data = request.args
        output_choice = 'display'
    command = perl_scripts + '/supported-organisms'
    if 'group' in data:
        command += ' -group ' + data['group']
    if 'format' in data:
        command += ' -format ' + data['format']
    if 'depth' in data:
        command += ' -depth ' + data['depth']
    if 'taxon' in data:
        command += ' -taxon ' + data['taxon']
    if 'unique_species' in data:
        command += ' -unique_species'
    if 'unique_genus' in data:
        command += ' -unique_genus'

    return util.run_command(command, output_choice, 'supported-organisms', 'txt','')

#### random_seq
@app.route('/random-seq', methods = ['POST', 'GET'])
def get_random_seq():
    output_choice = 'email'
    if request.method == 'POST':
        data = request.get_json(force=True) or request.form
    elif request.method == 'GET':
        data = request.args
        output_choice = 'display'
    command = perl_scripts + '/random-seq'

    if 'h' in data: # help message
        command += ' -h '
    if 'help' in data: # list options
        command += ' -help '
    if 'l' in data: # sequence length
        command += ' -l '  + data['l']
    if 'n' in data: # number of sequences
        command += ' -n '  + data['n']

    return util.run_command(command, output_choice, 'random-seq', 'fasta','')

################################################################
### Get information about polymorphic variations
@app.route('/variation-info', methods=['POST','GET'])
def get_variation_info():
    files = ''
    output_choice = 'display'
    if request.method == 'POST':
        data = request.form or request.get_json(force=True)
        files = request.files
    elif request.method == 'GET':
        data = request.args

    command = perl_scripts + '/variation-info'
    if 'h' in data: # help message and list options
        command += ' -h'
    mandatory_parameters = ['species','assembly']
    optional_parameters = ['species_suffix','release','format','type','col']
    default_param_values = {'format':'id'}
    fileupload_parameters = ['i']

    ## Read regular parameters
    parameters = util.read_parameters(data, mandatory_parameters, optional_parameters, default_param_values)
    if parameters['error'] != 0:
        return parameters['error_message']
    command += parameters['arguments']

    ## Upload input file if specified
    input_files = util.read_fileupload_parameters(data, files, fileupload_parameters, 'variation-info', '', True, ',')
    if input_files['error'] != 0:
        return input_files['error_message']
    command += input_files['arguments']
	
    return util.run_command(command, output_choice, 'variation-info', 'varBed', '' ,parameters['content_type'])

### retrieve variation sequence
@app.route('/retrieve-variation-seq', methods = ['POST', 'GET'])
def retrieve_variation_seq():
    files = ''
    output_choice = 'display'
    if request.method == 'POST':
        data = request.form or request.get_json(force=True)
        files = request.files
    elif request.method == 'GET':
        data = request.args

    command = rsat_bin + '/retrieve-variation-seq'
    if 'h' in data: # help message and list options
        command += ' -h'
    mandatory_parameters = ['species','assembly']
    optional_parameters = ['species_suffix','release','format','mml','col']
    default_param_values = {'format':'varBed', 'mml':'30', 'col':'1'}
    fileupload_parameters = ['i']

    ## Read regular parameters
    parameters = util.read_parameters(data, mandatory_parameters, optional_parameters, default_param_values)
    if parameters['error'] != 0:
        return parameters['error_message']
    command += parameters['arguments']

    ## Upload input file if specified
    input_files = util.read_fileupload_parameters(data, files, fileupload_parameters, 'retrieve-variation-seq', '')
    if input_files['error'] != 0:
        return input_files['error_message']
    command += input_files['arguments']

    return util.run_command(command, output_choice, 'retrieve-variation-seq', 'varSeq', '' ,parameters['content_type'])


################################################################
### variation-scan
@app.route('/variation-scan', methods=['POST','GET'])
def variation_scan():
    output_choice = 'display'
    files = ''
    if request.method == 'POST':
        data = request.form or request.get_json(force=True)
        files = request.files
    elif request.method == 'GET':
        data = request.args
    command = rsat_bin + '/variation-scan'

    if 'h' in data: # help message and list options
        command += ' -h'
    mandatory_parameters = ['m_format','bg']
    optional_parameters = ['top_matrices','mml','top_variation','lth','uth','calc_distrib','distrib_dir']
    default_param_values = {}
    fileupload_parameters = ['i','m']
    ## Read regular parameters
    parameters = util.read_parameters(data, mandatory_parameters, optional_parameters, default_param_values)
    if parameters['error'] != 0:
        return parameters['error_message']
    command += parameters['arguments']
    ## Upload input file if specified
    tmp_dir = util.make_tmp_dir('variation_scan')
    input_files = util.read_fileupload_parameters(data, files, fileupload_parameters, 'variation_scan', tmp_dir)
    if input_files['error'] != 0:
        return input_files['error_message']
    command += input_files['arguments']
	
    return util.run_command(command, output_choice, 'variation_scan', 'fasta', tmp_dir,parameters['content_type'])
    
#### fetch_sequence
#### return: {'output' : error, 'command' : command, 'server' : result file URL}
@app.route('/fetch-sequences', methods = ['POST', 'GET'])
def get_sequences():
    output_choice = 'email'
    files = ''
    if request.method == 'POST':
        data = request.form or request.get_json(force=True)
        files = request.files
    elif request.method == 'GET':
        data = request.args
        output_choice = 'display'
    command = perl_scripts + '/fetch-sequences'

    if 'h' in data: # help message and list options
        command += ' -h'

    mandatory_parameters = ['genome']
    optional_parameters = ['u','header_format','upstr_ext','downstr_ext','extend','reference','top','chunk']
    default_param_values = {'header_format':'UCSC'}
    fileupload_parameters = ['i']

    ## Read regular parameters
    parameters = util.read_parameters(data, mandatory_parameters, optional_parameters, default_param_values)
    if parameters['error'] != 0:
        return parameters['error_message']
    command += parameters['arguments']
    ## Upload input file if specified
    tmp_dir = util.make_tmp_dir('fetch-sequences')
    input_files = util.read_fileupload_parameters(data, files, fileupload_parameters, 'fetch-sequences', tmp_dir)
    if input_files['error'] != 0:
        if not 'u' in data:
            return input_files['error_message']
    else:
        if not 'u' in data:
            command += input_files['arguments']
	
    return util.run_command(command, output_choice, 'fetch-sequences', 'fasta', tmp_dir,parameters['content_type'])

    
### peak-motifs
### return: {'output' : errors/warnings, 'command' : command, 'server' : synthesis file URL}
@app.route('/peak-motifs', methods=['POST', 'GET'])
def peak_motifs():
    if request.method == 'POST':
        data = request.form or request.get_json(force=True)
        files = request.files
    elif request.method == 'GET':
        data = request.args
        
    command = perl_scripts + '/peak-motifs'
	### get parameters
    mandatory_parameters = []
    optional_parameters = ['max_seq_len','noov','str','title','image_format','soure','task','disco'
 'nmotifs','top_peaks','minol','maxol','markov','min_markov','max_markov','ci','r_plot']
    default_param_values = {'disco':'oligos,positions','task':'purge,seqlen,composition,disco,merge_motifs,split_motifs,motifs_vs_motifs,timelog,archive,synthesis,small_summary,motifs_vs_db,scan'}
    fileupload_parameters = ['i']
    optional_fileupload_parameters = ['ctrl','ref_motifs']
    ### Regular parameters
    parameters = util.read_parameters(data, mandatory_parameters, optional_parameters, default_param_values)
    if parameters['error'] != 0:
        return parameters['error_message']
    command += parameters['arguments']
    
    if 'motif_db' in data:
        motif_db = data['motif_db'].replace("'","")
        motif_db = motif_db.replace('"','')
        dbs = motif_db.split(",")
        matrix_db = util.supported_motif_database()
        for db in dbs:
            if db in matrix_db:
                command += " -motif_db " + matrix_db[db]['name'] + " " + matrix_db[db]['format'] + " " + public_html + "/motif_databases/" + matrix_db[db]['file']

    ## Upload input file if specified
    dir = util.make_tmp_dir("peak-motifs")
    input_files = util.read_fileupload_parameters(data, files, fileupload_parameters, 'peak-motifs', dir)
    if input_files['error'] != 0:
        return input_files['error_message']
    command += input_files['arguments']
    ## Upload optional file if specified
    #control file
    opt_files = util.read_fileupload_parameters(data, files, optional_fileupload_parameters, 'peak-motifs', dir, mandatory=False)
    command += opt_files['arguments']
		
	# outdir
    command += " -outdir '" + dir + "' -prefix peak-motifs &"
    result = ''
    os.system(command)
	#path to synthesis page	
    server_dir = dir + "/peak-motifs_synthesis.html"
	## change path to url
    server_dir = util.make_url(server_dir)
    return jsonify( {'output' : result, 'command' : command, 'server' : server_dir} )

if __name__ == '__main__':
    app.run(debug = True)
